/* -*-c++-*- OpenSceneGraph - Copyright (C) 2012 ADIT
*
* This library is open source and may be redistributed and/or modified under
* the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
* (at your option) any later version.  The full license is in LICENSE file
* included with this distribution, and on the openscenegraph.org website.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* OpenSceneGraph Public License for more details.
*/

/* Note, the code of GraphicsWindowFB was influenced by GraphicsWindowX11 class
* The GraphicsWindowX11 is licensed under OSGPL as above, with
* Copyright (C) 1998-2006 Robert Osfield.
*/

// TODO:
// - Make screen resolution configurable

#include <osgViewer/api/Wayland/GraphicsWindowWayland>
#ifdef WAYLAND_EGL_PBUFFER_SUPPORTED
#  include <osgViewer/api/Wayland/PixelBufferWayland>
#endif
#include <poll.h>
#include <osg/DeleteHandler>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <linux/types.h>

#include <algorithm>
#include  <iostream>

#include <wayland-egl.h>
#include <EGL/egl.h>
#include <EGL/eglext.h>

#include <wayland-client.h>
#include <linux/input.h>

#include "WLClientContext.hpp"
#include "WLWindowController.hpp"
#include "WLSurfaceEvent.hpp"

#define WL_UNUSED(A) (A)=(A)

static float mx;
static float my;


class WaylandKeyboardMap
{
public:

    WaylandKeyboardMap()
    {
        _extendedKeymap[KEY_ESC          ] = osgGA::GUIEventAdapter::KEY_Escape;
        _extendedKeymap[KEY_MINUS        ] = osgGA::GUIEventAdapter::KEY_Minus;
        _extendedKeymap[KEY_EQUAL        ] = osgGA::GUIEventAdapter::KEY_Equals;
        _extendedKeymap[KEY_BACKSPACE    ] = osgGA::GUIEventAdapter::KEY_BackSpace;
        _extendedKeymap[KEY_TAB          ] = osgGA::GUIEventAdapter::KEY_Tab;
        _extendedKeymap[KEY_BACKSLASH    ] = osgGA::GUIEventAdapter::KEY_Backslash;
        _extendedKeymap[KEY_CAPSLOCK     ] = osgGA::GUIEventAdapter::KEY_Caps_Lock;
        _extendedKeymap[KEY_SEMICOLON    ] = osgGA::GUIEventAdapter::KEY_Semicolon;
        _extendedKeymap[KEY_APOSTROPHE   ] = osgGA::GUIEventAdapter::KEY_Caret;
        _extendedKeymap[KEY_ENTER        ] = osgGA::GUIEventAdapter::KEY_Return;
        _extendedKeymap[KEY_COMMA        ] = osgGA::GUIEventAdapter::KEY_Comma;
        _extendedKeymap[KEY_DOT          ] = osgGA::GUIEventAdapter::KEY_Period;
        _extendedKeymap[KEY_SLASH        ] = osgGA::GUIEventAdapter::KEY_Slash;
        _extendedKeymap[KEY_SPACE        ] = osgGA::GUIEventAdapter::KEY_Space;
        _extendedKeymap[KEY_LEFTSHIFT    ] = osgGA::GUIEventAdapter::KEY_Shift_L;
        _extendedKeymap[KEY_RIGHTSHIFT   ] = osgGA::GUIEventAdapter::KEY_Shift_R;
        _extendedKeymap[KEY_LEFTCTRL     ] = osgGA::GUIEventAdapter::KEY_Control_L;
        _extendedKeymap[KEY_RIGHTCTRL    ] = osgGA::GUIEventAdapter::KEY_Control_R;
        _extendedKeymap[KEY_LEFTMETA     ] = osgGA::GUIEventAdapter::KEY_Meta_L;
        _extendedKeymap[KEY_RIGHTMETA    ] = osgGA::GUIEventAdapter::KEY_Meta_R;
        _extendedKeymap[KEY_LEFTALT      ] = osgGA::GUIEventAdapter::KEY_Alt_L;
        _extendedKeymap[KEY_RIGHTALT     ] = osgGA::GUIEventAdapter::KEY_Alt_R;
        _extendedKeymap[KEY_MENU         ] = osgGA::GUIEventAdapter::KEY_Menu;
        _extendedKeymap[KEY_PRINT        ] = osgGA::GUIEventAdapter::KEY_Print;
        _extendedKeymap[KEY_SCROLLLOCK   ] = osgGA::GUIEventAdapter::KEY_Scroll_Lock;
        _extendedKeymap[KEY_PAUSE        ] = osgGA::GUIEventAdapter::KEY_Pause;
        _extendedKeymap[KEY_HOME         ] = osgGA::GUIEventAdapter::KEY_Home;
        _extendedKeymap[KEY_PAGEUP       ] = osgGA::GUIEventAdapter::KEY_Page_Up;
        _extendedKeymap[KEY_END          ] = osgGA::GUIEventAdapter::KEY_End;
        _extendedKeymap[KEY_PAGEDOWN     ] = osgGA::GUIEventAdapter::KEY_Page_Down;
        _extendedKeymap[KEY_DELETE       ] = osgGA::GUIEventAdapter::KEY_Delete;
        _extendedKeymap[KEY_INSERT       ] = osgGA::GUIEventAdapter::KEY_Insert;
        _extendedKeymap[KEY_LEFT         ] = osgGA::GUIEventAdapter::KEY_Left;
        _extendedKeymap[KEY_UP           ] = osgGA::GUIEventAdapter::KEY_Up;
        _extendedKeymap[KEY_RIGHT        ] = osgGA::GUIEventAdapter::KEY_Right;
        _extendedKeymap[KEY_DOWN         ] = osgGA::GUIEventAdapter::KEY_Down;
        _extendedKeymap[KEY_NUMLOCK      ] = osgGA::GUIEventAdapter::KEY_Num_Lock;
        _extendedKeymap[KEY_KPSLASH      ] = osgGA::GUIEventAdapter::KEY_KP_Divide;
        _extendedKeymap[KEY_KPASTERISK   ] = osgGA::GUIEventAdapter::KEY_KP_Multiply;
        _extendedKeymap[KEY_KPMINUS      ] = osgGA::GUIEventAdapter::KEY_KP_Subtract;
        _extendedKeymap[KEY_KPPLUS       ] = osgGA::GUIEventAdapter::KEY_KP_Add;
        _extendedKeymap[KEY_KP7          ] = osgGA::GUIEventAdapter::KEY_KP_7;
        _extendedKeymap[KEY_KP8          ] = osgGA::GUIEventAdapter::KEY_KP_8;
        _extendedKeymap[KEY_KP9          ] = osgGA::GUIEventAdapter::KEY_KP_9;
        _extendedKeymap[KEY_KP4          ] = osgGA::GUIEventAdapter::KEY_KP_4;
        _extendedKeymap[KEY_KP5          ] = osgGA::GUIEventAdapter::KEY_KP_5;
        _extendedKeymap[KEY_KP6          ] = osgGA::GUIEventAdapter::KEY_KP_6;
        _extendedKeymap[KEY_KP1          ] = osgGA::GUIEventAdapter::KEY_KP_1;
        _extendedKeymap[KEY_KP2          ] = osgGA::GUIEventAdapter::KEY_KP_2;
        _extendedKeymap[KEY_KP3          ] = osgGA::GUIEventAdapter::KEY_KP_3;
        _extendedKeymap[KEY_KP0          ] = osgGA::GUIEventAdapter::KEY_KP_0;
        _extendedKeymap[KEY_KPDOT        ] = osgGA::GUIEventAdapter::KEY_KP_Separator;
        _extendedKeymap[KEY_KPENTER      ] = osgGA::GUIEventAdapter::KEY_KP_Enter;
        _extendedKeymap[KEY_KPDOT        ] = osgGA::GUIEventAdapter::KEY_KP_Decimal;

        _standardKeymap[KEY_A            ] = 'A';
        _standardKeymap[KEY_B            ] = 'B';
        _standardKeymap[KEY_C            ] = 'C';
        _standardKeymap[KEY_D            ] = 'D';
        _standardKeymap[KEY_E            ] = 'E';
        _standardKeymap[KEY_F            ] = 'F';
        _standardKeymap[KEY_G            ] = 'G';
        _standardKeymap[KEY_H            ] = 'H';
        _standardKeymap[KEY_I            ] = 'I';
        _standardKeymap[KEY_J            ] = 'J';
        _standardKeymap[KEY_K            ] = 'K';
        _standardKeymap[KEY_L            ] = 'L';
        _standardKeymap[KEY_M            ] = 'M';
        _standardKeymap[KEY_N            ] = 'N';
        _standardKeymap[KEY_O            ] = 'O';
        _standardKeymap[KEY_P            ] = 'P';
        _standardKeymap[KEY_Q            ] = 'Q';
        _standardKeymap[KEY_R            ] = 'R';
        _standardKeymap[KEY_S            ] = 'S';
        _standardKeymap[KEY_T            ] = 'T';
        _standardKeymap[KEY_U            ] = 'U';
        _standardKeymap[KEY_V            ] = 'V';
        _standardKeymap[KEY_W            ] = 'W';
        _standardKeymap[KEY_X            ] = 'X';
        _standardKeymap[KEY_Y            ] = 'Y';
        _standardKeymap[KEY_Z            ] = 'Z';
    }

    ~WaylandKeyboardMap() {}

    int remapKey(int key)
    {
        KeyMap::iterator itr = _extendedKeymap.find(key);
        if (itr != _extendedKeymap.end()) return itr->second;

        itr = _standardKeymap.find(key);
        if (itr != _standardKeymap.end()) return itr->second;

        return key;
    }

    bool remapExtendedKey(int& key)
    {
        KeyMap::iterator itr = _extendedKeymap.find(key);
        if (itr != _extendedKeymap.end())
        {
            key = itr->second;
            return true;
        }
        else return false;
    }

protected:

    typedef std::map<int, int> KeyMap;
    KeyMap _extendedKeymap;
    KeyMap _standardKeymap;
};

static int remapWaylandKey(int key)
{
    static WaylandKeyboardMap s_WaylandKeyboardMap;
    return s_WaylandKeyboardMap.remapKey(key);
}

namespace osgViewer
{
    class WLClientCtx : public osg::Referenced
    {
    public:
        WLClientCtx(const char* connectionName);
        ~WLClientCtx();

        WaylandBackend::WLClientContext* getWlContext()
        {
            return _wlCtx;
        }
    private:
        WaylandBackend::WLClientContext *_wlCtx;

    };

    class WaylandWindow : public WaylandBackend::WLSurfaceEvent
    {
    public:
        WaylandWindow(GraphicsWindowWayland*);
        int init(WaylandWindow* parentWLWindow);

        void* getWLDisplay()
        {
            return _wlClientCtx->getWlContext()->getDisplay();
        }

        WaylandBackend::WLWindowController* getWLWindowCtrl()
        {
            return _wlWinCtrl;
        }

        WLClientCtx* getWlClientCtx()
        {
            return _wlClientCtx.get();
        }

        int dispatchEvents(uint32_t timeout);

        ~WaylandWindow();
    private:
        osg::ref_ptr<WLClientCtx>              _wlClientCtx;
        WaylandBackend::WLWindowController*    _wlWinCtrl;
        GraphicsWindowWayland*                 _osgGraWindow;
        int createWLContext();

        virtual void keyboardEnter(void* seat, uint32_t serial,
                                   struct wl_array* keys);

        virtual void keyboardKeymap(void* seat, uint32_t format, int fd,
                                    uint32_t size);

        virtual void keyboardModifiers(void* seat, uint32_t serial,
                           uint32_t mods_depressed, uint32_t mods_latched,
                           uint32_t mods_locked, uint32_t group);

        virtual void keyboardKey(void* seat, uint32_t serial, uint32_t time,
                                uint32_t key, uint32_t state_w);

        virtual void keyboardLeave(void* seat, uint32_t serial);

        virtual void keyboardRepeatInfo(void* seat, int32_t rate,
                                        int32_t delay);

        virtual void pointerEnter(void* seat, uint32_t serial, double sx,
                                  double sy);

        virtual void pointerMotion(void* seat, uint32_t time, double sx,
                                   double sy);

        virtual void pointerButton(void* seat, uint32_t serial, uint32_t time,
                                  uint32_t button, uint32_t state);

        virtual void pointerAxis(void* seat, uint32_t time, uint32_t axis,
                                 int value);

        virtual void pointerLeave(void* seat, uint32_t serial);


        virtual void touchDown(void* seat, uint32_t serial, uint32_t time,
                               int32_t id, double xw, double yw);

        virtual void touchMotion(void* seat, uint32_t time, int32_t id,
                                 double xw, double yw);

        virtual void touchFrame(void* seat);

        virtual void touchCancel(void* seat);

        virtual void touchUp(void* seat, uint32_t serial, uint32_t time, int32_t id);
    };
}

using namespace osgViewer;

WLClientCtx::WLClientCtx(const char* connectionName)
{
    _wlCtx = new WaylandBackend::WLClientContext(connectionName);
}

WLClientCtx::~WLClientCtx()
{
    if (NULL != _wlCtx)
    {
        delete _wlCtx;
    }
}

WLEglDisplay::WLEglDisplay(struct wl_display* wl_display)
{
    EGLint eglMajorVersion, eglMinorVersion;
    _eglDisplay = eglGetDisplay(wl_display);
    if (EGL_NO_DISPLAY != _eglDisplay)
    {
        if (!eglInitialize(_eglDisplay, &eglMajorVersion, &eglMinorVersion))
        {
            OSG_NOTICE << "failed to initialize egl display" << std::endl;
        }
        else
        {
            OSG_NOTICE << "egl initialized: major version = " << eglMajorVersion
                    << " minor version = " << eglMinorVersion << std::endl;
        }
    }
}

EGLDisplay WLEglDisplay::getEglDisplay()
{
    return _eglDisplay;
}

WLEglDisplay::~WLEglDisplay()
{
    if (EGL_NO_DISPLAY != _eglDisplay)
        eglTerminate(_eglDisplay);
}



WaylandWindow::WaylandWindow(GraphicsWindowWayland* osgGraphicsWindow)
:_wlClientCtx(),
 _wlWinCtrl(NULL),
_osgGraWindow(osgGraphicsWindow)
{

}

WaylandWindow::~WaylandWindow()
{
    if (NULL != _wlWinCtrl)
    {
        _wlClientCtx->getWlContext()->destroyWindowController(_wlWinCtrl);
    }
    _wlClientCtx = NULL;
}

int WaylandWindow::init(WaylandWindow* parentWLWindow)
{
    int ret = 0;
    if (NULL == parentWLWindow)
    {
        ret = createWLContext();
    }
    else
    {
        _wlClientCtx = parentWLWindow->getWlClientCtx();
        if (_wlClientCtx.get() == NULL)
            ret = -1;
    }

    if (ret >= 0)
    {
        _wlWinCtrl =  _wlClientCtx->getWlContext()->
                       createWindowController(this);
        if (NULL == _wlWinCtrl)
        {
            OSG_NOTICE << "Failed to create wayland surface" << std::endl;
            ret = -1;
        }
    }
    return ret;
}


int
WaylandWindow::dispatchEvents(uint32_t timeout)
{
    return _wlClientCtx->getWlContext()->dispatchEvents(timeout);
}

int WaylandWindow::createWLContext()
{
    int ret = -1;
    _wlClientCtx = new WLClientCtx(NULL);

    if (_wlClientCtx.get() != NULL)
    {
        if (_wlClientCtx->getWlContext() != NULL)
        {
            if (_wlClientCtx->getWlContext()->getDisplay() != NULL)
            {
                ret = 0;
            }
            else
            {
                OSG_NOTICE<<"wl_display_connect failed"<<std::endl;
            }
        }
        else
        {
            OSG_NOTICE << "Failed to allocate for WLClientContext" << std::endl;
        }
    }
    else
    {
        OSG_NOTICE << "Failed to allocate memory WLClientCtx" << std::endl;
    }

    return ret;
}


void
WaylandWindow::keyboardEnter(void* seat, uint32_t serial,
                             struct wl_array* keys)
{
    WL_UNUSED(serial);
    WL_UNUSED(keys);
    WL_UNUSED(seat);
    OSG_DEBUG << "keyboardEnter" << std::endl;
}

void
WaylandWindow::keyboardKeymap(void* seat, uint32_t format,
                              int fd, uint32_t size)
{
    WL_UNUSED(format);
    WL_UNUSED(fd);
    WL_UNUSED(size);
    OSG_DEBUG << "keyboardKeymap" << std::endl;
}

void
WaylandWindow::keyboardModifiers(void* seat, uint32_t serial,
                                 uint32_t mods_depressed,
                                 uint32_t mods_latched, uint32_t mods_locked,
                                 uint32_t group)
{
    WL_UNUSED(serial);
    WL_UNUSED(mods_depressed);
    WL_UNUSED(mods_latched);
    WL_UNUSED(mods_locked);
    WL_UNUSED(group);
    WL_UNUSED(seat);
    OSG_DEBUG << "keyboardModifiers" << std::endl;
}

void
WaylandWindow::keyboardKey(void* seat, uint32_t serial, uint32_t time,
                           uint32_t key, uint32_t state_w)
{
    WL_UNUSED(serial);
    WL_UNUSED(time);
    WL_UNUSED(seat);
    if (state_w == WL_KEYBOARD_KEY_STATE_RELEASED)
    {
        OSG_DEBUG<<"KeyboardHandleKey RELEASED serial"<< serial<<" time "<< time<<" key "<<key<<" remapped "<<remapWaylandKey(key)<<" state "<<state_w<<std::endl;
        _osgGraWindow->getEventQueue()->keyRelease(remapWaylandKey(key));
    }
    else
    {
        OSG_DEBUG<<"KeyboardHandleKey PRESS serial"<< serial<<" time "<< time<<" key "<<key<<" remapped "<<remapWaylandKey(key)<<" state "<<state_w<<std::endl;
        _osgGraWindow->getEventQueue()->keyPress(remapWaylandKey(key));
    }
}

void
WaylandWindow::keyboardLeave(void* seat, uint32_t serial)
{
    WL_UNUSED(serial);
    WL_UNUSED(seat);
    OSG_DEBUG << "keyboardLeave" << std::endl;
}

void
WaylandWindow::keyboardRepeatInfo(void* seat, int32_t rate, int32_t delay)
{
    WL_UNUSED(seat);
    WL_UNUSED(rate);
    WL_UNUSED(delay);
    OSG_DEBUG << "keyboardRepeatInfo" << std::endl;
}

void
WaylandWindow::pointerEnter(void* seat, uint32_t serial, double sx, double sy)
{
    WL_UNUSED(seat);
    WL_UNUSED(serial);
    OSG_DEBUG<<"pointerEnter: Coords " << sx << ":" << sy << std::endl;
}

void WaylandWindow::pointerMotion(void* seat, uint32_t time, double sx,
                                  double sy)
{
    WL_UNUSED(time);
    WL_UNUSED(seat);
    mx = sx;
    my = sy;
    _osgGraWindow->transformMouseXY(mx, my);
    _osgGraWindow->getEventQueue()->mouseMotion(mx, my);
    OSG_DEBUG<<"pointerMotion x: "<< mx << " y: " << my << std::endl;
}

void WaylandWindow::pointerButton(void* seat,  uint32_t serial, uint32_t time,
                                  uint32_t button, uint32_t state)
{
    WL_UNUSED(serial);
    WL_UNUSED(time);
    WL_UNUSED(seat);
    if(button >= BTN_LEFT)
    {
        if (button == BTN_TOUCH)
            button = BTN_LEFT;

        if (state == WL_POINTER_BUTTON_STATE_RELEASED)
        {
            OSG_DEBUG<<"pointerButton:RELEASED button:"<<(button + 1) - BTN_LEFT<<std::endl;
            _osgGraWindow->getEventQueue()->mouseButtonRelease(mx, my, (button + 1) - BTN_LEFT);
        }
        else
        {
            OSG_DEBUG<<"pointerButton:PRESS button:"<<(button + 1) - BTN_LEFT<<std::endl;
            _osgGraWindow->getEventQueue()->mouseButtonPress(mx, my, (button + 1) - BTN_LEFT);
        }
    }
}

void WaylandWindow::pointerAxis(void* seat, uint32_t time, uint32_t axis,
                                int value)
{
    WL_UNUSED(time);
    WL_UNUSED(axis);
    WL_UNUSED(seat);
    if (value > 0)
        _osgGraWindow->getEventQueue()->mouseScroll(osgGA::GUIEventAdapter::SCROLL_UP);
    else
        _osgGraWindow->getEventQueue()->mouseScroll(osgGA::GUIEventAdapter::SCROLL_DOWN);
    OSG_DEBUG << "pointerAxis"<< std::endl;
}

void WaylandWindow::pointerLeave(void* seat, uint32_t serial)
{
    WL_UNUSED(serial);
    WL_UNUSED(seat);
    OSG_DEBUG << "pointerLeave" << std::endl;
}

void
WaylandWindow::touchDown(void* seat, uint32_t serial,
        uint32_t time, int32_t id, double x, double y)
{
    WL_UNUSED(serial);
    WL_UNUSED(seat);

    _osgGraWindow->handleTouchDown(time, id, x, y);
}

void
WaylandWindow::touchUp(void* seat, uint32_t serial, uint32_t time, int32_t id)
{
    WL_UNUSED(serial);
    WL_UNUSED(seat);

    _osgGraWindow->handleTouchUp(time, id);
}

void
WaylandWindow::touchMotion(void* seat, uint32_t time, int32_t id,
                           double x, double y)
{
    WL_UNUSED(seat);
    _osgGraWindow->handleTouchMotion(time, id, x, y);
}

void
WaylandWindow::touchFrame(void* seat)
{
    WL_UNUSED(seat);
    OSG_DEBUG<<"touchFrame"<<std::endl;
}

void
WaylandWindow::touchCancel(void* seat)
{
    WL_UNUSED(seat);
    OSG_DEBUG<<"touchCancel"<<std::endl;
}


namespace {
/**
 * @brief The Helper class for init to clean up on error case without using goto
 */
class CloseGuard {
public:
    CloseGuard(GraphicsWindowWayland *gw) : _gw(gw) {}
    ~CloseGuard() { if (_gw != 0) { _gw->closeImplementation(); } }
    void release() { _gw = 0; }

private:
    GraphicsWindowWayland* _gw;
};
}

GraphicsWindowWayland::~GraphicsWindowWayland()
{
    close(false);
}

bool
GraphicsWindowWayland::checkEGLError(const char* str)
{
    EGLint err = eglGetError();
    if (err != EGL_SUCCESS)
    {
        OSG_WARN<<"Warning: "<<str<<" EGL error "<<std::hex<<err<<std::dec<< std::endl;
        return true;
    }
    return false;
}

void GraphicsWindowWayland::destroyWindow()
{
    if (_eglSurface != EGL_NO_SURFACE)
    {
        eglDestroySurface(_wlEglDisplay->getEglDisplay(), _eglSurface);
    }
    if (NULL != _wl_native_window)
    {
        wl_egl_window_destroy(_wl_native_window);
    }
}

int GraphicsWindowWayland::createWindow(int width, int height, EGLConfig eglConfig)
{
    int ret = -1;

    _wl_native_window = wl_egl_window_create((struct wl_surface*)_wlWindow->
                                             getWLWindowCtrl()->getWlSurface(),
                                             _width, _height);

    if (_wl_native_window != NULL)
    {
        _eglSurface = eglCreateWindowSurface(_wlEglDisplay->getEglDisplay(), eglConfig,
                (EGLNativeWindowType)_wl_native_window, NULL);

        if (_eglSurface != EGL_NO_SURFACE)
        {
            ret = 0;
        }
        else
        {
            OSG_FATAL
            << "GraphicsWindowWayland::init() - eglCreateWindowSurface(..) failed."
            << std::endl;
        }
    }
    else
    {
        OSG_NOTICE<<"wl_egl_window_create failed"<<std::endl;
    }
    return ret;
}

void GraphicsWindowWayland::getAttributes(Attributes *attrs, Traits *ctxTraits)
{
    attrs->push_back(EGL_RED_SIZE);
    attrs->push_back(ctxTraits->red);
    attrs->push_back(EGL_GREEN_SIZE);
    attrs->push_back(ctxTraits->green);
    attrs->push_back(EGL_BLUE_SIZE);
    attrs->push_back(ctxTraits->blue);
    attrs->push_back(EGL_DEPTH_SIZE);
    attrs->push_back(ctxTraits->depth);

    if (ctxTraits->alpha)
    {
        attrs->push_back(EGL_ALPHA_SIZE);
        attrs->push_back(ctxTraits->alpha);
    }
    if (ctxTraits->stencil)
    {
        attrs->push_back(EGL_STENCIL_SIZE);
        attrs->push_back(ctxTraits->stencil);
    }

    if (ctxTraits->sampleBuffers)
    {
        attrs->push_back(EGL_SAMPLE_BUFFERS);
        attrs->push_back(ctxTraits->sampleBuffers);
    }
    if (ctxTraits->samples)
    {
        attrs->push_back(EGL_SAMPLES);
        attrs->push_back(ctxTraits->samples);
    }

    attrs->push_back(EGL_RENDERABLE_TYPE);

#if defined(OSG_GLES2_AVAILABLE) && !defined(OSG_GLES3_AVAILABLE)
    attrs->push_back(EGL_OPENGL_ES2_BIT);
#elif defined(OSG_GLES3_AVAILABLE)
    attrs->push_back(EGL_OPENGL_ES3_BIT_KHR);
#else
    attrs->push_back(EGL_OPENGL_ES_BIT);
#endif

    attrs->push_back(EGL_NONE);
    attrs->push_back(EGL_NONE);
}

void
GraphicsWindowWayland::init()
{
    OSG_DEBUG << "GraphicsWindowsWayland::init" << std::endl;
    int ret = 0;
    EGLConfig eglConfig = 0;
    int numConfigs;
    EGLint contextAttribs[] =
#ifdef OSG_GLES3_AVAILABLE
    { EGL_CONTEXT_CLIENT_VERSION, 3, EGL_NONE };
#else
    { EGL_CONTEXT_CLIENT_VERSION, 2, EGL_NONE };
#endif
    Attributes attributes;
    GraphicsWindowWayland* osgWlWindow = NULL;
    WaylandWindow* parentWlWindow = NULL;
    unsigned int init_stage;
    WaylandBackend::WLWindowController* winCtrl = NULL;
    bool visible = true;


    if (_initialized)
        return;

    if (!_traits)
    {
        _valid = false;
        return;
    }

    _width  = _traits->width;
    _height = _traits->height;

    // Will call this->closeImplementation() on destroy if not reset
    CloseGuard guard(this);

    WindowData* windowData = _traits->inheritedWindowData ? dynamic_cast<WindowData*>(_traits->inheritedWindowData.get()) : NULL;
    if (windowData)
    {
        _layerId = windowData->_layerID;
        _surfaceId = windowData->_surfaceID;
    }

    EGLContext contextToShareWith = EGL_NO_CONTEXT;
    {
        osg::ref_ptr<GraphicsContext> sharedContext;
        if (_traits.valid() && _traits->sharedContext.lock(sharedContext))
        {
            osgWlWindow = dynamic_cast<GraphicsWindowWayland*>(sharedContext.get());
            if ((osgWlWindow) && (osgWlWindow->valid()))
            {
                contextToShareWith = osgWlWindow->getContext();
                if (EGL_NO_CONTEXT != contextToShareWith)
                {
                    parentWlWindow = osgWlWindow->getWLWindow();
                    _wlEglDisplay = osgWlWindow->getWLEglDisplay();
                }
            }
        }
    }

    for (init_stage = 0; (!_initialized) && (ret >= 0); init_stage++)
    {
        switch (init_stage)
        {
        case 0:
            ret = -1;
            _wlWindow = new WaylandWindow(this);
            if (NULL != _wlWindow)
            {
                ret = _wlWindow->init(parentWlWindow);
            }
            break;
        case 1:
            if (NULL ==_wlEglDisplay.get())
            {
                _wlEglDisplay = new WLEglDisplay((struct wl_display*)
                                                  _wlWindow->getWLDisplay());
                if (NULL == _wlEglDisplay.get())
                {
                    OSG_NOTICE << "Failed to allocate memory for WLEglDisplay object" << std::endl;
                    ret = -1;
                }
            }
            eglBindAPI( EGL_OPENGL_ES_API);
            break;
        case 2:
            getAttributes(&attributes, _traits.get());

            if (!eglChooseConfig(_wlEglDisplay->getEglDisplay(), &(attributes.front()),
                                &eglConfig, 1, &numConfigs) || (numConfigs != 1))
            {
                OSG_FATAL << "GraphicsWindowWayland::init() - eglChooseConfig() failed."
                << std::endl;
                ret = -1;
            }
            break;
        case 3:
            ret = createWindow(_width, _height, eglConfig);
            break;
        case 4:
            _context = eglCreateContext(_wlEglDisplay->getEglDisplay(), eglConfig,
                                        contextToShareWith, contextAttribs);
            OSG_NOTICE << "GraphicsWindowWayland::init() " << _context << " context will share with " << contextToShareWith << std::endl;
            if (_context == EGL_NO_CONTEXT)
            {
                OSG_FATAL << "GraphicsWindowWayland::init() - eglCreateContext(..) failed. Error: " << eglGetError()
                << std::endl;
                ret = -1;
            }
            break;
        case 5:
            eglMakeCurrent(_wlEglDisplay->getEglDisplay(), _eglSurface,
                           _eglSurface, _context);

            eglSwapInterval(_wlEglDisplay->getEglDisplay(), 1);
            if (eglGetError() != EGL_SUCCESS)
            {
                OSG_NOTICE<<"eglMakeCurrent failed"<<std::endl;
                ret = -1;
            }
            eglMakeCurrent(_wlEglDisplay->getEglDisplay(), EGL_NO_SURFACE,
                            EGL_NO_SURFACE, EGL_NO_CONTEXT);
            break;
        case 6:
            winCtrl = _wlWindow->getWLWindowCtrl();

            ret = winCtrl->init(_surfaceId, _layerId, _width, _height);

            if (ret < 0)
                OSG_NOTICE << "Failed to initialize WL window for "
                           << winCtrl->getShellName()
                           << " error=" << ret << std::endl;

            break;
        case 7:
            ret = winCtrl->Configure(_traits->x, _traits->y, _width, _height);

            if (-ENOTSUP == ret)
                OSG_DEBUG << winCtrl->getShellName()
                          << " does not support Configure and placement of window"
                          << std::endl;

            if (windowData != NULL)
                visible = windowData->_surfaceVisibility;

            ret = winCtrl->setVisibility(visible);
            if (-ENOTSUP == ret) {
                OSG_DEBUG << winCtrl->getShellName()
                          << " does not support setting visibility" << std::endl;
                ret = 0;
            }

            break;
        default:
            _initialized = true;
            _valid = true;
            break;
        }
    }

    checkEGLError("after eglCreateContext()");
    _successfulInitStage = init_stage - 1;
    if (_initialized)
    {
        _successfulInitStage--; //include the default case
        // Everything went fine, no need to cleanup
        guard.release();
    }
}

bool
GraphicsWindowWayland::realizeImplementation()
{
    if (_realized)
    {
        OSG_NOTICE
        << "GraphicsWindowWayland::realizeImplementation() Already realized"
        << std::endl;
        return true;
    }

    if (!_initialized)
    init();

    if (!_initialized)
    return false;

    _realized = true;

    return true;
}

bool
GraphicsWindowWayland::makeCurrentImplementation()
{
    if (!_realized)
    {
        OSG_NOTICE
        << "Warning: GraphicsWindow not realized, cannot do makeCurrent."
        << std::endl;
        return false;
    }

    bool result = eglMakeCurrent(_wlEglDisplay->getEglDisplay(), _eglSurface,
                                 _eglSurface, _context) == EGL_TRUE;
    checkEGLError("after eglMakeCurrent()");
    return result;
}

bool
GraphicsWindowWayland::releaseContextImplementation()
{
    if (!_realized)
    {
        OSG_NOTICE
        << "Warning: GraphicsWindow not realized, cannot do release context."
        << std::endl;
        return false;
    }

    bool result = eglMakeCurrent(_wlEglDisplay->getEglDisplay(), EGL_NO_SURFACE,
                                 EGL_NO_SURFACE, EGL_NO_CONTEXT) == EGL_TRUE;

    checkEGLError("after eglMakeCurrent() release");
    return result;
}

void
GraphicsWindowWayland::closeImplementation()
{
    bool deinitComplete = false;
    int deinit_stage;
    _initialized = false;
    _realized = false;
    _valid = false;

    for (deinit_stage = _successfulInitStage; !deinitComplete; deinit_stage--)
    {
        switch (deinit_stage)
        {
        case 7:
            //no need to reset surface configuration
            break;
        case 6:
            //There is nothing to de-initialize
            break;
        case 5:
            eglMakeCurrent(_wlEglDisplay->getEglDisplay(), EGL_NO_SURFACE,
                            EGL_NO_SURFACE, EGL_NO_CONTEXT);

            if (eglGetError() != EGL_SUCCESS)
            {
                OSG_NOTICE<<"eglMakeCurrent failed"<<std::endl;
            }
            break;
        case 4:
            if (_context)
            {
                eglDestroyContext(_wlEglDisplay->getEglDisplay(), _context);
                _context = 0;
            }
            break;
        case 3:
            destroyWindow();
            break;
        case 2:
            //Not required to destroy config as it is just a number
            break;
        case 1:
            //This object is ref counted and will be destroyed automatically
            //if no one is using it
            _wlEglDisplay = NULL;
            break;
        case 0:
            if (NULL != _wlWindow)
                delete _wlWindow;
            break;
        default:
            deinitComplete = true;
            break;

        }
    }
}

void
GraphicsWindowWayland::swapBuffersImplementation()
{
    if (!_realized)
    return;

    eglSwapBuffers(getEGLDisplay(), _eglSurface);
    checkEGLError("after eglSwapBuffers()");
}

bool
GraphicsWindowWayland::setWindowRectangleImplementation(int x, int y, int width, int height)
{
    int ret = 0;

    if (!_realized)
    {
        OSG_FATAL
        << "GraphicsWindowWayland::setWindowRectangleImplementation() GraphicsWindow not realized"
        << std::endl;
        return false;
    }

    eglWaitNative(EGL_CORE_NATIVE_ENGINE);
    checkEGLError("after eglWaitNative()");

    wl_egl_window_resize(_wl_native_window, width, height, x, y);

    ret = _wlWindow->getWLWindowCtrl()->ConfigureSynchronous(x, y, width,
                                                             height);

    if (-ENOTSUP == ret)
    {
        OSG_DEBUG << _wlWindow->getWLWindowCtrl()->getShellName()
                  << " does not support synchronous resizing, falling back to asynchronous"
                  << std::endl;
        ret = _wlWindow->getWLWindowCtrl()->Configure(x, y, width, height);
    }

    if (-ENOTSUP == ret)
    {
        OSG_DEBUG << _wlWindow->getWLWindowCtrl()->getShellName()
                  << " does not support setting window size from client side"
                  << std::endl;
    }
    else if (ret < 0)
    {
        OSG_NOTICE << _wlWindow->getWLWindowCtrl()->getShellName()
                   << " failed to configure window size, error = " << ret
                   << std::endl;
    }
    else
    {
        _width = width;
        _height = height;
    }

    return (ret >= 0);
}

void
GraphicsWindowWayland::setWindowName(const std::string& winName)
{
    int ret;
    if(_initialized)
    {
        _traits->windowName = winName;
        ret = _wlWindow->getWLWindowCtrl()->setTitle(winName.c_str());

        if (ret == -ENOTSUP)
        {
            OSG_DEBUG << _wlWindow->getWLWindowCtrl()->getShellName()
                      << " does not support setting window title"
                      << std::endl;
        }
        else if (ret < 0)
        {
            OSG_NOTICE << _wlWindow->getWLWindowCtrl()->getShellName()
                       << " failed to set window title, error = " << ret
                       << std::endl;
        }
    }
}

void
GraphicsWindowWayland::raiseWindow()
{
    int ret;
    if(_initialized)
    {
        ret = _wlWindow->getWLWindowCtrl()->setTopLevel();

        if (ret == -ENOTSUP)
        {
            OSG_DEBUG << _wlWindow->getWLWindowCtrl()->getShellName()
                      << " does not support raise of window to top level"
                      << std::endl;
        }
        else if (ret < 0)
        {
            OSG_NOTICE << _wlWindow->getWLWindowCtrl()->getShellName()
                       << " failed to set window to top, error = " << ret
                       << std::endl;
        }
    }
}

bool
GraphicsWindowWayland::checkEvents()
{
    bool retVal = false;
    int numEvents = 0;

    numEvents = _wlWindow->dispatchEvents(0);
    if (numEvents == 0)
        retVal = GraphicsWindow::checkEvents();
    else if (numEvents > 0)
            retVal = true;

    return retVal;
}

void
GraphicsWindowWayland::handleTouchDown(uint32_t time, int32_t id, float x, float y)
{
    WL_UNUSED(time);

    if (true == _initialized)
    {
        transformMouseXY(x, y);
        _lastTouchPoints[id].set(x, y);
        osg::ref_ptr<osgGA::GUIEventAdapter> touchEvent =
        getEventQueue()->touchBegan(id,osgGA::GUIEventAdapter::TOUCH_BEGAN, x, y);

        addKnownTouchPoints(touchEvent.get(), id);

        _currentTouchContacts.set(id);
    }

    OSG_DEBUG<<"TouchHandleDown"<<std::endl;
}

void
GraphicsWindowWayland::handleTouchUp(uint32_t time, int32_t id)
{
    WL_UNUSED(time);

    if (true == _initialized)
    {
        _currentTouchContacts.reset(id);

        osg::ref_ptr<osgGA::GUIEventAdapter> touchEvent =
        getEventQueue()->touchEnded(id,osgGA::GUIEventAdapter::TOUCH_ENDED,
        _lastTouchPoints[id].x(), _lastTouchPoints[id].y(), 0);

        addKnownTouchPoints (touchEvent.get(), id);
    }

    OSG_DEBUG<<"TouchHandleUp"<<std::endl;
}

void
GraphicsWindowWayland::handleTouchMotion(uint32_t time, int32_t id, float x, float y)
{
    WL_UNUSED(time);

    if (true == _initialized)
    {
        transformMouseXY(x, y);
        _lastTouchPoints[id].set(x, y);

        osg::ref_ptr<osgGA::GUIEventAdapter> touchEvent =
        getEventQueue()->touchMoved(id,  osgGA::GUIEventAdapter::TOUCH_MOVED, x, y);

        addKnownTouchPoints (touchEvent.get(), id);

        // Just to make sure;
        _currentTouchContacts.set(id);
    }
    OSG_DEBUG<<"TouchHandleMotion"<<std::endl;
}

void GraphicsWindowWayland::addKnownTouchPoints(osgGA::GUIEventAdapter *touchEvent, uint32_t skip)
{
    for (unsigned int i = 0; i < _currentTouchContacts.size(); i++)
    {
        if (i == skip || not _currentTouchContacts.test(i))
        continue;

        touchEvent->addTouchPoint(i, osgGA::GUIEventAdapter::TOUCH_STATIONERY,
        _lastTouchPoints[i].x(), _lastTouchPoints[i].y());
    }
}

void GraphicsWindowWayland::transformMouseXY(float& x, float& y )
{
    if (getEventQueue()->getUseFixedMouseInputRange())
    {
        osgGA::GUIEventAdapter* eventState = getEventQueue()->getCurrentEventState();

        x = eventState->getXmin() + (eventState->getXmax()-eventState->getXmin())*x/float(getTraits()->width);
        y = eventState->getYmin() + (eventState->getYmax()-eventState->getYmin())*y/float(getTraits()->height);
    }
}

bool
GraphicsWindowWayland::setSwapInterval(int intervalFrames) const
{
    return (EGL_TRUE == eglSwapInterval(_wlEglDisplay->getEglDisplay(),
                             intervalFrames));
}

void GraphicsWindowWayland::setSurfaceVisibility(uint32_t surfaceVisibility)
{
    int ret;
    if (!_realized)
    {
        OSG_FATAL
        << "GraphicsWindowWayland::setSurfaceVisibility() GraphicsWindow not realized"
        << std::endl;
        return;
    }
    ret = _wlWindow->getWLWindowCtrl()->setVisibility((surfaceVisibility > 0));
    if (-ENOTSUP == ret)
    {
        OSG_DEBUG << _wlWindow->getWLWindowCtrl()->getShellName()
                  << " does not support visibility setting"<<std::endl;
    }
    else if (ret < 0)
    {
        OSG_WARN << _wlWindow->getWLWindowCtrl()->getShellName()
                 << " failed to set visibility, error = " << ret << std::endl;
    }
}

class WaylandWindowingSystemInterface : public osg::GraphicsContext::WindowingSystemInterface
{
protected:
    bool _errorHandlerSet;

public:
    WaylandWindowingSystemInterface()
    {
    }

    ~WaylandWindowingSystemInterface()
    {
        if (osg::Referenced::getDeleteHandler())
        {
            osg::Referenced::getDeleteHandler()->setNumFramesToRetainObjects(0);
            osg::Referenced::getDeleteHandler()->flushAll();
        }
    }

    virtual unsigned int
    getNumScreens(const osg::GraphicsContext::ScreenIdentifier& si)
    {
        return 1;
    }

    virtual void
    getScreenSettings(const osg::GraphicsContext::ScreenIdentifier& si,
    osg::GraphicsContext::ScreenSettings & resolution)
    {
        resolution.width = 1024;
        resolution.height = 768;
        resolution.colorDepth = 24;
        resolution.refreshRate = 0;
    }

    virtual bool
    setScreenSettings(const osg::GraphicsContext::ScreenIdentifier& si,
    const osg::GraphicsContext::ScreenSettings & resolution)
    {
        return false;
    }

    virtual void
    enumerateScreenSettings(const osg::GraphicsContext::ScreenIdentifier& si,
    osg::GraphicsContext::ScreenSettingsList & resolutionList)
    {
        OSG_NOTICE
        << "WaylandWindowingSystemInterface::enumerateScreenSettings() not supported."
        << std::endl;
    }

    virtual osg::GraphicsContext*
    createGraphicsContext(osg::GraphicsContext::Traits* traits)
    {
        if (traits->pbuffer)
        {
#ifdef WAYLAND_EGL_PBUFFER_SUPPORTED
            osg::ref_ptr<osgViewer::PixelBufferWayland> pbuffer = new PixelBufferWayland(traits);
            if (pbuffer->valid()) return pbuffer.release();
            else return 0;
#else
            osg::ref_ptr<osgViewer::GraphicsWindowWayland> window = new GraphicsWindowWayland(traits);
            if (window->valid()) return window.release();
            else return 0;
#endif
        }
        else
        {
            osg::ref_ptr<osgViewer::GraphicsWindowWayland> window = new GraphicsWindowWayland(traits);
            if (window->valid()) return window.release();
            else return 0;
        }
    }

};

struct RegisterWindowingSystemInterfaceProxy
{
    RegisterWindowingSystemInterfaceProxy()
    {
        osg::GraphicsContext::setWindowingSystemInterface(new WaylandWindowingSystemInterface);
    }

    ~RegisterWindowingSystemInterfaceProxy()
    {
        if (osg::Referenced::getDeleteHandler())
        {
            osg::Referenced::getDeleteHandler()->setNumFramesToRetainObjects(0);
            osg::Referenced::getDeleteHandler()->flushAll();
        }

        osg::GraphicsContext::setWindowingSystemInterface(0);
    }
};

RegisterWindowingSystemInterfaceProxy createWindowingSystemInterfaceProxy;

// declare C entry point for static compilation.
extern "C" void
graphicswindow_Wayland(void)
{
    osg::GraphicsContext::setWindowingSystemInterface(new WaylandWindowingSystemInterface);
}
